//
//  PhotoAttachmentViewController.swift
//  IFS
//
//  Created by 曹云霄 on 2018/1/5.
//  Copyright © 2018年 上海勾芒信息科技有限公司. All rights reserved.
//

import UIKit
import ImagePicker
import SKPhotoBrowser
import RxCocoa
import RxSwift
import SnapKit


/// 更新高度后回调
protocol PhotoAttachmentDelegate {
    func updatePhotoAttachment(_ height: CGFloat, _ key: String?)
}

class PhotoAttachmentViewController: BaseViewController {

    /// RxSwift自动释放
    let disposeBag = DisposeBag()
    
    /// 更新图片附件
    var delegate: PhotoAttachmentDelegate?
    
    /// PhotoAttachmentViewModel
    lazy final var attachmentViewModel: PhotoAttachmentViewModel = {
        var attachmentViewModel = PhotoAttachmentViewModel()
        return attachmentViewModel
    }()
    
    /// item 尺寸
    var size: CGSize?
    /// key
    var key: String?
    /// 间隔
    let SPACE: Int = 5
    /// 每行个数
    var COUNT: Int = 2
    /// 图片数量默认最多五张
    var imageCount: Int = 5
    
    /// 图片高度
    lazy final var imageHeight: CGFloat = {
        // 30表示UICollectionView距左右边距
        var imageHeight = ((kWidth - 30) - CGFloat((COUNT + kONE) * SPACE)) / CGFloat(COUNT) * 3 / 4
        return imageHeight
    }()
    
    /// 默认section高度
    let sectionHeight: CGFloat! = 30
    
    /// 附件
    @IBOutlet weak var collectionViewFlowLayout: UICollectionViewFlowLayout!
    @IBOutlet weak var photoAttachmentCollectionView: UICollectionView!
    
    override func viewDidLoad() {
        super.viewDidLoad()

        setupCollectionView()
    }
    
    /// 设置item Size
    open func setItemSize(_ size: CGSize, _ count: Int) {
        self.size = size
        self.imageHeight = size.height
        self.COUNT = count
        if let collectionViewFlowLayout = collectionViewFlowLayout {
            collectionViewFlowLayout.itemSize = size
            photoAttachmentCollectionView.reloadData()
        }
    }
    
    // MARK: - 设置图片附件
    fileprivate func setupCollectionView() {
        collectionViewFlowLayout.minimumLineSpacing = CGFloat(SPACE)
        collectionViewFlowLayout.minimumInteritemSpacing = CGFloat(SPACE)
        collectionViewFlowLayout.sectionInset = UIEdgeInsetsMake(CGFloat(SPACE), CGFloat(SPACE), CGFloat(SPACE), CGFloat(SPACE))
        if let size = size {
            collectionViewFlowLayout.itemSize = size
        }else {
            collectionViewFlowLayout.itemSize = CGSize(width: imageHeight * 4 / 3, height: imageHeight)
        }
    }
    
    // MARK: - 计算图片附件所需高度
    open func attachmentHeight(_ count: Int, _ key: String?) ->CGFloat {
        self.key = key
        let rows = (count + COUNT - kONE) / COUNT
        let height = CGFloat(rows) * imageHeight + CGFloat((rows - kONE) * SPACE)
        if height < 20 {
            return 20
        }
        return height
    }
}

// MARK: - 图片附件
extension PhotoAttachmentViewController: UICollectionViewDelegate,UICollectionViewDataSource {
    func collectionView(_ collectionView: UICollectionView, numberOfItemsInSection section: Int) -> Int {
        return attachmentViewModel.photoAttachments.count
    }

    func collectionView(_ collectionView: UICollectionView, cellForItemAt indexPath: IndexPath) -> UICollectionViewCell {
        let cell =  attachmentViewModel.dequeueReusableCell(PhotoAttachmentCollectionViewCell.name(), indexPath, collectionView) as! PhotoAttachmentCollectionViewCell
        cell.delegate = self
        return cell
    }
    
    func collectionView(_ collectionView: UICollectionView, didSelectItemAt indexPath: IndexPath) {
        let attachment = attachmentViewModel.photoAttachments[indexPath.item]
        if attachment.attachmentType == .ATTACHMENT_ADD {
            let imagePickerController = ImagePickerController()
            imagePickerController.delegate = self
            let count = attachmentViewModel.photoAttachments.count
            if count - kONE >= imageCount {
                ShowMessage(String(format: "最多只能拍摄%d张照片", imageCount))
                return
            }else {
                imagePickerController.imageLimit = imageCount - (count - kONE)
            }
            present(imagePickerController, animated: true, completion: nil)
        }else {
            var images = [SKPhoto]()
            for attachment in attachmentViewModel.photoAttachments {
                if attachment.attachmentType == .ATTACHMENT_ADD { continue }
                var photo: SKPhoto!
                switch attachment.attachmentType {
                case .ATTACHMENT_HTTP:
                    photo = SKPhoto.photoWithImageURL(attachment.attachment as! String)
                    break
                case .ATTACHMENT_LOCAL:
                    photo = SKPhoto.photoWithImage(attachment.attachment as! UIImage)
                    break
                default:
                    break
                }
                images.append(photo)
                photo.shouldCachePhotoURLImage = false
            }
            let browser = SKPhotoBrowser(photos: images)
            browser.initializePageIndex(indexPath.item)
            present(browser, animated: true, completion: {})
        }
    }
}

// MARK: - 拍照
extension PhotoAttachmentViewController: ImagePickerDelegate {
    func wrapperDidPress(_ imagePicker: ImagePickerController, images: [UIImage]) {
        for image in images {
            var images = [SKPhoto]()
            let photo = SKPhoto.photoWithImage(image)
            images.append(photo)
            let browser = SKPhotoBrowser(photos: images)
            browser.initializePageIndex(0)
            present(browser, animated: true, completion: {})
        }
    }
    
    func doneButtonDidPress(_ imagePicker: ImagePickerController, images: [UIImage]) {
        for image in images {
            attachmentViewModel.photoAttachments.insert(PhotoAttachmentModel(image, .ATTACHMENT_LOCAL), at: kZERO)
            dismiss(animated: true, completion: {[weak self] ()->() in
                let count = self?.attachmentViewModel.photoAttachments.count
                let height = self?.attachmentHeight(count!,self?.key)
                DispatchQueue.main.async(execute: {
                    self?.photoAttachmentCollectionView.reloadData()
                    self?.delegate?.updatePhotoAttachment(height!,self?.key)
                })
            })
        }
    }
    
    func cancelButtonDidPress(_ imagePicker: ImagePickerController) {
        dismiss(animated: true, completion: nil)
    }
}

// MARK: - 删除附件
extension PhotoAttachmentViewController: DeleteAttachmentDelegate {
    func deleteAttachment(_ indexPath: IndexPath) {
        ShowAlertView(hint, "确定删除此附件吗？", [cancel,sure], .actionSheet) {[weak self] (index) in
            if index == kZERO { return }
            self?.attachmentViewModel.photoAttachments.remove(at: indexPath.item)
            let count = self?.attachmentViewModel.photoAttachments.count
            let height = self?.attachmentHeight(count!,self?.key)
            DispatchQueue.main.async(execute: {
                self?.photoAttachmentCollectionView.reloadData()
                self?.delegate?.updatePhotoAttachment(height!,self?.key)
            })
        }
    }
}

