package com.wmdigit.cateringservicedemo.adapter;

import android.content.Context;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import android.widget.AdapterView;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.RecyclerView;

import com.wmdigit.cateringservicedemo.R;
import com.wmdigit.cateringservicedemo.model.ApiButton;

import java.util.List;


/**
 * ApiButton的适配器类，继承自RecyclerView.Adapter并泛型化以 ViewHolder 为持有器类型
 * 该适配器用于将ApiButton对象列表绑定到RecyclerView上显示
 * @author dizi
 */
public class ApiButtonAdapter extends RecyclerView.Adapter<ApiButtonAdapter.ViewHolder> {

    /**
     * 上下文
     */
    private Context context;
    /**
     * ApiButton对象列表，RecyclerView将展示这些数据
     */
    private List<ApiButton> data;
    /**
     * 列表点击监听
     */
    private OnItemClickListener onItemClickListener;

    /**
     * 构造函数，初始化上下文和数据列表
     *
     * @param context 上下文对象，用于访问资源和创建视图
     * @param data    ApiButton对象列表，RecyclerView将展示这些数据
     */
    public ApiButtonAdapter(Context context, List<ApiButton> data) {
        this.context = context;
        this.data = data;
    }

    /**
     * 设置列表项的点击事件监听器
     *
     * @param onItemClickListener 列表项点击事件的监听接口实现者
     *                             该接口实现者负责处理列表项被点击时的行为
     */
    public void setOnItemClickListener(OnItemClickListener onItemClickListener) {
        this.onItemClickListener = onItemClickListener;
    }


    /**
     * 创建并返回一个新的ViewHolder，用于加载和缓存视图
     *
     * @param parent   父容器，用于确定ViewHolder的根视图
     * @param viewType 视图类型，影响视图的布局方式
     * @return 新创建的ViewHolder实例
     */
    @NonNull
    @Override
    public ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        // 使用LayoutInflater从资源文件中加载视图
        View itemView = LayoutInflater.from(context).inflate(R.layout.item_api_button, parent, false);
        // 返回持有该视图的ViewHolder实例
        return new ViewHolder(itemView, onItemClickListener);
    }

    /**
     * 将数据绑定到指定位置的ViewHolder上
     *
     * @param holder   要绑定数据的ViewHolder
     * @param position 绑定数据的列表项位置
     */
    @Override
    public void onBindViewHolder(@NonNull ViewHolder holder, int position) {
        // 将指定位置的数据绑定到ViewHolder
        holder.bind(data.get(position));
    }

    /**
     * 返回数据列表的项数
     *
     * @return 数据列表的大小，即RecyclerView中应显示的项数
     */
    @Override
    public int getItemCount() {
        // 返回数据列表的大小
        return data.size();
    }

    /**
     * 表示RecyclerView中的条目视图的持有者
     * 用于优化性能，避免在每次滚动时频繁创建和销毁视图
     */
    public static class ViewHolder extends RecyclerView.ViewHolder {

        /**
         * 声明ViewHolder中所需的三个TextView对象
         */
        private TextView tvNo, tvName, tvDesc;


        /**
         * 构造函数，初始化ViewHolder并为其设置点击事件监听
         *
         * @param itemView 父视图项，用于查找和配置视图组件
         * @param listener 点击事件监听器，如果提供，将为 itemView 设置点击事件
         */
        public ViewHolder(@NonNull View itemView, OnItemClickListener listener) {
            super(itemView);
            // 初始化视图组件，这些组件显示接口的编号、方法名和描述
            tvNo = itemView.findViewById(R.id.tv_interface_number);
            tvName = itemView.findViewById(R.id.tv_interface_method);
            tvDesc = itemView.findViewById(R.id.tv_interface_desc);

            // 如果提供了点击事件监听器，则为 itemView 设置点击事件监听器
            if (listener != null){
                itemView.setOnClickListener(v -> {
                    // 获取当前 ViewHolder 在 RecyclerView 适配器中的位置
                    int position = getAdapterPosition();
                    // 如果位置有效，则调用点击事件监听器的回调方法
                    if (position != RecyclerView.NO_POSITION) {
                        listener.onItemClick(v, position);
                    }
                });
            }
        }

        /**
         * 将ViewHolder中的视图元素与特定的ApiButton数据绑定
         *
         * @param apiButton 包含要显示的数据的ApiButton对象
         */
        public void bind(ApiButton apiButton){
            // 将ApiButton对象的数据设置到缓存的TextView中
            tvNo.setText(apiButton.getMethodNo());
            tvName.setText(apiButton.getMethodName());
            tvDesc.setText(apiButton.getMethodDesc());
        }
    }

    /**
     * 点击事件监听器接口
     * 用于处理列表项或视图项被点击时的事件
     */
    public interface OnItemClickListener{
        /**
         * 当列表项或视图项被点击时调用
         *
         * @param view 点击事件发生的视图
         * @param position 点击事件发生的项的位置
         */
        void onItemClick(View view, int position);
    }
}
