package com.wmdigit.cateringservicedemo;

import androidx.appcompat.app.AppCompatActivity;
import androidx.recyclerview.widget.GridLayoutManager;
import androidx.viewbinding.ViewBinding;

import android.content.ComponentName;
import android.os.Bundle;
import android.os.IBinder;
import android.os.RemoteException;
import android.text.method.ScrollingMovementMethod;

import com.wmdigit.cateringservicedemo.adapter.ApiButtonAdapter;
import com.wmdigit.cateringservicedemo.data.Constant;
import com.wmdigit.cateringservicedemo.data.ErrorCode;
import com.wmdigit.cateringservicedemo.databinding.ActivityMainBinding;
import com.wmdigit.cateringservicedemo.model.ApiButton;
import com.wmdigit.cateringservicedemo.utils.DateUtils;
import com.wmdigit.common.model.ProductsDTO;
import com.wmdigit.service.IOnDetectionListener;
import com.wmdigit.service.IOnInitListener;
import com.wmdigit.service.WmSdk;
import com.wmdigit.service.aidl.model.DetectResult;
import com.wmdigit.service.listener.OnServiceConnectListener;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * @author dizi
 */
public class MainActivity extends AppCompatActivity {

    private ActivityMainBinding viewBinding;
    private ApiButtonAdapter mainProcessAdapter, statusInfoAdapter, otherAdapter;

    /**
     * 是否在识别时生成图片并返回
     * 如无需展示该图片，建议关闭可以提升运算速度
     * DEMO中设置为TRUE仅为了展示效果
     */
    private boolean generateBitmap = true;

    private ExecutorService executorService = Executors.newSingleThreadExecutor();

    /**
     * 识别结果回调
     */
    private IOnDetectionListener.Stub detectionListener = new IOnDetectionListener.Stub() {

        /**
         * 当识别操作完成时被调用
         *
         * @param detectResult 识别结果，包含识别出的商品信息等
         * @throws RemoteException 如果与系统通信时发生错误
         */
        @Override
        public void onDetected(DetectResult detectResult) throws RemoteException {
            showDetectResult(detectResult);
        }

    };

    private void showDetectResult(DetectResult detectResult) {
        // 检查识别结果是否为空，如果为空则记录异常并返回
        if (detectResult == null){
            log("识别结果异常");
            return;
        }

        // 如果识别结果的错误码不表示成功，则记录失败信息并返回
        if (detectResult.getCode() != ErrorCode.ERROR_CODE_SUCCESS){
            log("识别失败，错误码：" + detectResult.getCode());
            log("错误原因：" + ErrorCode.getErrorMessage(detectResult.getCode()));
            return;
        }

        // 检查识别出的商品码列表，如果没有商品码则记录未识别出商品的信息并返回
        if(detectResult.getProductCodes() == null || detectResult.getProductCodes().size() == 0){
            log("未识别出商品");
            return;
        }

        // 将识别出的商品码列表转换为字符串并记录
        String result = detectResult.getProductCodes().toString();
        log("识别结果：" + result);

        // 如果识别过程生成的图片不为空且未被回收，则在用户界面上展示该图片
        if (detectResult.getBitmap() != null && !detectResult.getBitmap().isRecycled()) {
            viewBinding.ivOutput.post(() -> viewBinding.ivOutput.setImageBitmap(detectResult.getBitmap()));
        }
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        // 初始化视图界面
        initView();
        // 初始化数据
        initData();
        // 初始化监听器
        initListener();
        // 连接服务
        connectService();
    }

    @Override
    protected void onStart() {
        super.onStart();
        // 接口3.5 注册识别结果回调
        // 注意当MainActivity初次启动时，服务未完成连接，该方法不会生效，需要在服务连接成功后，再次调用该接口注册回调
        WmSdk.getInstance().registerDetectionListener(generateBitmap, detectionListener);
    }

    @Override
    protected void onStop() {
        super.onStop();
        // 接口3.6 反注册识别结果回调
        WmSdk.getInstance().unregisterDetectionListener();
    }

    /**
     * 连接绑定服务程序
     */
    private void connectService() {
        // 接口4.1 判断服务程序是否安装
        if (WmSdk.getInstance().checkServiceAppInstalled(this)) {
            // 接口3.1 绑定服务
            WmSdk.getInstance().bindService(this, new OnServiceConnectListener() {
                @Override
                public void onConnected(ComponentName componentName, IBinder iBinder) {
                    log("服务绑定成功, 准备初始化");
                    // 接口3.3 初始化学习数据
                    WmSdk.getInstance().init(new IOnInitListener.Stub() {
                        @Override
                        public void onComplete() throws RemoteException {
                            log("学习数据初始化完成");
                            runOnUiThread(() -> {
                                // 接口3.5 注册识别结果回调
                                WmSdk.getInstance().registerDetectionListener(generateBitmap, detectionListener);
                            });
                        }
                    });
                }

                @Override
                public void onDisconnected(ComponentName componentName) {
                    log("服务已断开");
                }
            });
        }

    }

    /**
     * 初始化用户界面视图
     *
     * 此方法负责设置活动的布局视图，配置RecyclerView的布局管理器，并初始化可滚动的文本视图
     * 它主要关注于设置主流程接口、状态接口和其他接口的RecyclerView为网格布局，以及设置日志文本视图的滚动方法
     */
    private void initView() {
        // 使用ActivityMainBinding.inflate方法来绑定活动的布局视图
        viewBinding = ActivityMainBinding.inflate(getLayoutInflater());
        // 设置活动的内容视图为绑定的布局视图的根元素
        setContentView(viewBinding.getRoot());
        // 为rvMainProcessInterface RecyclerView设置GridLayoutManager，定义网格布局的列数为4
        viewBinding.rvMainProcessInterface.setLayoutManager(new GridLayoutManager(this, 4));
        // 为rvStatusInterface RecyclerView设置GridLayoutManager，定义网格布局的列数为4
        viewBinding.rvStatusInterface.setLayoutManager(new GridLayoutManager(this, 4));
        // 为rvOtherInterface RecyclerView设置GridLayoutManager，定义网格布局的列数为4
        viewBinding.rvOtherInterface.setLayoutManager(new GridLayoutManager(this, 4));
        // 设置tvLog TextView的滚动方法，允许文本内容垂直滚动
        viewBinding.tvLog.setMovementMethod(ScrollingMovementMethod.getInstance());
    }


    /**
     * 初始化数据并设置适配器
     *
     * 该方法负责初始化三个不同的适配器并为各自的RecyclerView设置适配器
     * 它们分别是mainProcessAdapter, statusInfoAdapter和其他接口的适配器
     */
    private void initData() {
        // 初始化主流程数据适配器
        mainProcessAdapter = new ApiButtonAdapter(this, Constant.mainProcessData);
        // 为显示主流程接口的RecyclerView设置适配器
        viewBinding.rvMainProcessInterface.setAdapter(mainProcessAdapter);

        // 初始化状态信息数据适配器
        statusInfoAdapter = new ApiButtonAdapter(this, Constant.statusInfoData);
        // 为显示状态信息接口的RecyclerView设置适配器
        viewBinding.rvStatusInterface.setAdapter(statusInfoAdapter);

        // 初始化其他数据适配器
        otherAdapter = new ApiButtonAdapter(this, Constant.otherData);
        // 为显示其他接口的RecyclerView设置适配器
        viewBinding.rvOtherInterface.setAdapter(otherAdapter);
    }


    /**
     * 初始化各列表的点击事件监听器
     */
    private void initListener() {
        // 为mainProcessAdapter设置点击事件监听，当项被点击时调用mainProcessClickEvent方法
        mainProcessAdapter.setOnItemClickListener((view, position) -> {
            mainProcessClickEvent(Constant.mainProcessData.get(position));
        });

        // 为statusInfoAdapter设置点击事件监听，当项被点击时调用statusInfoClickEvent方法
        statusInfoAdapter.setOnItemClickListener((view, position) -> {
            statusInfoClickEvent(Constant.statusInfoData.get(position));
        });

        // 为otherAdapter设置点击事件监听，当项被点击时调用otherClickEvent方法
        otherAdapter.setOnItemClickListener((view, position) -> {
            otherClickEvent(Constant.otherData.get(position));
        });
    }

    /**
     * 主流程接口点击事件
     * @param apiButton
     */
    private void mainProcessClickEvent(ApiButton apiButton){
        switch (apiButton.getMethodNo()){
            case "3.1":
                log("服务绑定接口，已在onCreate()中实现");
                break;

            case "3.2":
                log("解绑服务程序接口，已在onDestroy()中实现");
                break;

            case "3.3":
                log("初始化学习数据接口，已在3.1接口的回调中实现");
                break;

            case "3.4":
                log("准备导入测试用商品资料");
                // 导入商品资料
                List<ProductsDTO> list = new ArrayList<>();
                list.add(new ProductsDTO("西红柿炒鸡蛋","000323", "XHSCJD", "9.80", 1));
                list.add(new ProductsDTO("红烧茄子","000324", "HSQZ", "8.80", 1));
                list.add(new ProductsDTO("套餐A","000325", "TCA", "22.80", 1));
                list.add(new ProductsDTO("套餐B","000326", "TCB", "15.80", 1));
                list.add(new ProductsDTO("套餐C","000327", "TCC", "18.80", 1));
                list.add(new ProductsDTO("套餐D","000328", "TCD", "30.00", 1));
                list.add(new ProductsDTO("套餐E","000329", "TCF", "23.80", 1));
                list.add(new ProductsDTO("披萨","000330", "PS", "25.00", 1));
                list.add(new ProductsDTO("牛角包","000331", "NJB", "6.40", 1));
                list.add(new ProductsDTO("三明治","000332", "SMZ", "16.80", 1));
                list.add(new ProductsDTO("菠萝包","000333", "BLB", "12.50", 1));
                list.add(new ProductsDTO("蛋挞","000334", "DT", "6.00", 1));
                list.add(new ProductsDTO("豆沙包","000335", "DSB", "6.60", 1));
                list.add(new ProductsDTO("麻花包","000336", "MHB", "9.20", 1));
                WmSdk.getInstance().importProducts(list);
                log("商品资料导入完成");
                break;

            case "3.5":
                log("注册识别结果回调，已在onStart()和3.3接口的回调中实现");
                break;

            case "3.6":
                log("反注册识别结果回调，已在onStop()中实现");
                break;

            case "3.7":
                log("准备重新识别");
                executorService.execute(()->{
                    // 子线程执行识别接口
                    DetectResult result = WmSdk.getInstance().autoDetect(generateBitmap);
                    showDetectResult(result);
                });
                break;

            default:
                break;

        }
    }

    /**
     * 状态信息点击事件
     * @param apiButton
     */
    private void statusInfoClickEvent(ApiButton apiButton){
        switch (apiButton.getMethodNo()){
            case "4.1":
                boolean isServiceAppInstalled = WmSdk.getInstance().checkServiceAppInstalled(this);
                log("获取服务程序安装状态：" + isServiceAppInstalled);
                break;

            case "4.2":
                boolean isServiceConnected = WmSdk.getInstance().checkServiceConnected();
                log("获取服务程序连接状态：" + isServiceConnected);
                break;

            case "4.3":
                boolean isActivated = WmSdk.getInstance().checkActivation();
                log("获取服务程序激活状态：" + isActivated);
                break;

            case "4.4":
                boolean isCameraCropped = WmSdk.getInstance().checkCameraCrop();
                log("获取摄像头裁剪状态：" + isCameraCropped);
                break;

            case "4.5":
                boolean isLearningDataInitCompleted = WmSdk.getInstance().checkLearningDataInitCompleted();
                log("获取学习数据初始化状态：" + isLearningDataInitCompleted);
                break;

            default:
                break;
        }
    }

    /**
     * 其他接口点击事件
     * @param apiButton
     */
    private void otherClickEvent(ApiButton apiButton){
        switch (apiButton.getMethodNo()){
            case "5.1":
                log("打开设置页");
                WmSdk.getInstance().openSettingPage();
                break;

            case "5.2":
                log("重置摄像头背景，请清空摄像头下的台面");
                WmSdk.getInstance().resetCameraBackground();
                log("重置成功");
                break;

            default:
                break;
        }
    }

    /**
     * 将消息记录到日志视图中
     *
     * @param msg 要记录的消息字符串
     */
    private void log(String msg) {
        runOnUiThread(()->{
            viewBinding.tvLog.append(DateUtils.getTodayTime() + " " + msg + "\n");
            // 将日志滑到底部
//            viewBinding.tvLog.scrollTo(0, viewBinding.tvLog.getBottom());
        });
    }

    /**
     * 清空日志显示
     *
     * 该方法的作用是将日志文本视图的显示内容清空，以便于重新开始记录日志或显示新的日志信息
     * 它通过设置文本视图的文本为空字符串来实现日志内容的清除
     */
    private void clearLog() {
        viewBinding.tvLog.setText("");
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        // 接口3.2 解绑服务
        WmSdk.getInstance().unbindService();
        // 释放视图绑定资源，避免内存泄漏
        viewBinding = null;
        // 清空主进程适配器，避免外部引用导致的问题
        mainProcessAdapter = null;
        // 清空状态信息适配器，防止内存泄漏
        statusInfoAdapter = null;
        // 清空其他适配器，统一资源管理
        otherAdapter = null;
    }

}